/*
 * ioboard interface library
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <linux/dimmio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <errno.h>

#include "ioboard.h"

/* dimmio data */
struct dimm_io button_io[3]; /* Button signal */
struct dimm_io portc_io[8]; /* Bargraph, LCD signal */
struct dimm_io latchc_io[2]; /* Latch control */
struct dimm_io lcdc_io[3]; /* LCD control */
int io1dev = -1;

int buttons_open = 0, bargraph_open = 0, latch_open = 0, lcd_open = 0;

/* Configures a pin by port and bit number */
void set_config_by_port_bit(struct dimm_io *pin, short port, short bit, unsigned char mode, unsigned char pull_up_dn) {
	int error;
  
	pin->port_number = port;
	pin->bit_number = bit;
	pin->mode = mode;
	pin->pull_up_dn = pull_up_dn;
  
	error = ioctl(io1dev, DIMM_IO_IOCTL_SET_CONFIG_BY_PORT_BIT, pin);
	if (error < 0) 
	{
		fprintf(stderr, "Unable to configure device. Error code: %i\n", errno);
		close(io1dev);
		exit(EXIT_FAILURE);
	}
}

/* Reads a pin using offset */
void read_by_offset(struct dimm_io *pin) {
	int error;

	lseek(io1dev, pin->offset, SEEK_SET);
	error = read(io1dev, &(pin->state), 1);

	if (error < 0) 
	{
		fprintf(stderr, "Error in reading from device.  Error code: %i\n", errno);
		close(io1dev);
		exit(EXIT_FAILURE);
	}
}

/* Writes a value to a pin using offset */
void write_by_offset(struct dimm_io *pin) {
	int error;
  
	lseek(io1dev, pin->offset, SEEK_SET);
	error = write(io1dev, &(pin->state), 1);

	if (error < 0) {
		fprintf(stderr, "Error in writing to device\n");
		close(io1dev);
		exit(EXIT_FAILURE);
	}
}

/* Read a series of pins */
void read_pins(struct dimm_io *pins, int num_pins) {
	int c, error;

	for (c = 0; c < num_pins; c++) {
		lseek(io1dev, pins[c].offset, SEEK_SET);
		error = read(io1dev, &(pins[c].state), 1);

		if (error < 0) {
			fprintf(stderr, "Error in writing to device\n");
			close(io1dev);
			exit(EXIT_FAILURE);
		}
	}
}

/* Write a series of pins */
void write_pins(struct dimm_io *pins, int num_pins) {
	int c, error;

	for (c = 0; c < num_pins; c++) {
		lseek(io1dev, pins[c].offset, SEEK_SET);
		error = write(io1dev, &(pins[c].state), 1);

		if (error < 0) {
			fprintf(stderr, "Error in writing to device\n");
			close(io1dev);
			exit(EXIT_FAILURE);
		}
	}
}

void open_io1dev(void) {
	if (io1dev > -1)
		return;

	/* Open dimmio device */
	io1dev = open("/dev/io1", O_RDWR);

	if (io1dev < 0) {
		fprintf(stderr, "Unable to open /dev/io1\n");
		exit(EXIT_FAILURE);
	}
}

void open_buttons(void) {
	int c, errorcode;

	if (buttons_open)
		return;

	open_io1dev();

	for (c = 0; c < 3; c++) {
		button_io[c].port_number = 10; /* Port K */
		button_io[c].bit_number = c + 4; /* Pins 4, 5, and 6 */
		button_io[c].mode = 0; /* Input */
		button_io[c].state = 0;
		button_io[c].pull_up_dn = 0; /* No idea */

		errorcode = ioctl(io1dev, DIMM_IO_IOCTL_SET_CONFIG_BY_PORT_BIT, &button_io[c]);

		if (errorcode < 0) {
			fprintf(stderr, "Unable to configure /dev/io1 for button %i. Error code %i\n", c, errorcode);
			close(io1dev);
			exit(EXIT_FAILURE);
		}
	}

	buttons_open = 1;
}

void open_latch(void) {
	int c, errorcode;

	if (latch_open)
		return;

	open_io1dev();

	latchc_io[0].bit_number = 0; /* Pin 0 */
	latchc_io[1].bit_number = 2; /* Pin 2 */

	for (c = 0; c < 2; c++) {
		latchc_io[c].port_number = 9; /* Port J */
		latchc_io[c].mode = 1; /* Output */
		latchc_io[c].state = 1;
		latchc_io[c].pull_up_dn = 0;

		errorcode = ioctl(io1dev, DIMM_IO_IOCTL_SET_CONFIG_BY_PORT_BIT, &latchc_io[c]);

		if (errorcode < 0) {
			fprintf(stderr, "Unable to configure /dev/io1. Error code %i\n", errorcode);
			close(io1dev);
			exit(EXIT_FAILURE);
		}
	}

	latch_open = 1;
}

void open_bargraph(void) {
	int c, errorcode;

	if (bargraph_open)
		return;

	open_io1dev();
	open_latch();

	/* LEDs */
	for (c = 0; c < 8; c++) {
		portc_io[c].port_number = 2; /* Port C */
		portc_io[c].bit_number = c; /* Pins 0 - 7 */
		portc_io[c].mode = 1; /* Output */
		portc_io[c].state = 0;
		portc_io[c].pull_up_dn = 0;

		errorcode = ioctl(io1dev, DIMM_IO_IOCTL_SET_CONFIG_BY_PORT_BIT, &portc_io[c]);

		if (errorcode < 0) {
			fprintf(stderr, "Unable to configure /dev/io1. Error code %i\n", errorcode);
			close(io1dev);
			exit(EXIT_FAILURE);
		}
	}

	bargraph_open = 1;
}

void lcd_execute(int wait_time) {
	lcdc_io[0].state = 1;
	write_by_offset(&lcdc_io[0]);

	usleep(wait_time);

	lcdc_io[0].state = 0;
	write_by_offset(&lcdc_io[0]);
}

void open_lcd(void) {
	int c, errorcode;

	if (lcd_open)
		return;

	open_io1dev();

	/*
	 * 0: K0 - E
	 * 1: D5 - RS
	 * 2: D6 - R/W
	 */
	lcdc_io[0].port_number = 10;	/* Port K */
	lcdc_io[0].bit_number = 0;	/* Pin 0 */
	lcdc_io[1].port_number = 3;	/* Port D */
	lcdc_io[1].bit_number = 5;	/* Pin 5 */
	lcdc_io[2].port_number = 3;	/* Port D */
	lcdc_io[2].bit_number = 6;	/* Pin 6 */

	/* LEDs */
	for (c = 0; c < 8; c++) {
		lcdc_io[c].mode = 1; /* Output */
		lcdc_io[c].state = 0;
		lcdc_io[c].pull_up_dn = 0;

		errorcode = ioctl(io1dev, DIMM_IO_IOCTL_SET_CONFIG_BY_PORT_BIT, &lcdc_io[c]);

		if (errorcode < 0) {
			fprintf(stderr, "Unable to configure /dev/io1. Error code %i\n", errorcode);
			close(io1dev);
			exit(EXIT_FAILURE);
		}
	}

	/* Set 8 bit data length, both lines, 5x7 font */
	/* lcd_io states are all 0 initially */
	portc_io[7].state = portc_io[6].state = portc_io[2].state = 0;
	portc_io[5].state = portc_io[4].state = portc_io[3].state = 1;
	write_pins(lcdc_io, 3);
	write_pins(portc_io, 8);

	/* Execute */
	lcd_execute(40);

	lcd_open = 1;
}

void get_buttons(int *b1, int *b2, int *b3) {
	int temp[3];

	/* Save the old state */
	temp[0] = button_io[0].state;
	temp[1] = button_io[1].state;
	temp[2] = button_io[2].state;
	
	/* Get the new state */
	read_pins(button_io, 3);

	/* If any have changed state since last time, debounce */
	if ((button_io[0].state != temp[0]) || (button_io[1].state != temp[1]) || (button_io[2].state != temp[2])) {
		usleep(1500);

		/* This is weird. If the state doesn't change during the debounce,
		   we keep that. If it does, we keep *that*, too. */
		read_pins(button_io, 3);
	}

	*b1 = button_io[0].state;
	*b2 = button_io[1].state;
	*b3 = button_io[2].state;
}

void enable_bargraph(int state) {
	if (state)
		latchc_io[1].state = 1;
	else
		latchc_io[1].state = 0;

	write_by_offset(&latchc_io[1]);
}

void set_bargraph(int bargraph[8]) {
	/* Unlatch bargraph */
	latchc_io[0].state = 0;
	write_by_offset(&latchc_io[0]);

	/* Set bargraph */
	portc_io[0].state = bargraph[0];
	portc_io[1].state = bargraph[1];
	portc_io[2].state = bargraph[2];
	portc_io[3].state = bargraph[3];
	portc_io[4].state = bargraph[4];
	portc_io[5].state = bargraph[5];
	portc_io[6].state = bargraph[6];
	portc_io[7].state = bargraph[7];
	write_pins(portc_io, 8);

	/* Latch bargraph */
	latchc_io[0].state = 1;
	write_by_offset(&latchc_io[0]);
}

void set_lcd(int display, int cursor_mode, int direction, int shift) {
	/*
	 * display: 0 - LCD off
	 *	    1 - LCD on
	 * cursor_mode: 0 - invisible
	 *		1 - visible, static
	 *		2 - visible, blinking
	 * direction: 0 - move left
	 *	      1 - move right
	 * shift: 0 - do not shift display on write
	 *	  1 - shift display on write
	 */

	/* Set Display mode */
	lcdc_io[1].state = lcdc_io[2].state = 0;
	portc_io[7].state = portc_io[6].state = portc_io[5].state = portc_io[4].state = 0;
	portc_io[3].state = 1;
	portc_io[2].state = display;
	portc_io[1].state = (cursor_mode > 0)?1:0;
	portc_io[0].state = (cursor_mode > 1)?1:0;
	write_pins(lcdc_io, 3);
	write_pins(portc_io, 8);
	lcd_execute(40);

	/* Set Entry mode */
	lcdc_io[1].state = lcdc_io[2].state = 0;
	portc_io[7].state = portc_io[6].state = portc_io[5].state = portc_io[4].state = portc_io[3].state = 0;
	portc_io[2].state = 1;
	portc_io[1].state = direction;
	portc_io[0].state = shift;
	write_pins(lcdc_io, 3);
	write_pins(portc_io, 8);
	lcd_execute(40);
}

void clear_lcd(void) {
	lcdc_io[1].state = lcdc_io[2].state = 0;
	portc_io[7].state = portc_io[6].state = portc_io[5].state = portc_io[4].state = \
		portc_io[3].state = portc_io[2].state = portc_io[1].state = 0;
	portc_io[0].state = 1;
	write_pins(lcdc_io, 3);
	write_pins(portc_io, 8);

	lcd_execute(1640); /* Wait 1.64 ms */
}

void set_lcd_cursor(unsigned int x, unsigned int y) {
	int addr, c, wait;

	if (x > 39)
		return;

	lcdc_io[1].state = lcdc_io[2].state = 0;

	if ((x == 0) && (y == 0)) {
		portc_io[7].state = portc_io[6].state = portc_io[5].state = \
			portc_io[4].state = portc_io[3].state = portc_io[2].state = 0;
		portc_io[1].state = 1;

		wait = 1640;
	}
	else {
		addr = x + y?0:0x40;

		portc_io[7].state = 1;
		for (c = 0; c < 7; c++)
			portc_io[c].state = (addr >> c) & 1;

		wait = 40;
	}

	write_pins(lcdc_io, 3);
	write_pins(portc_io, 8);

	lcd_execute(wait);
}

void shift_lcd(int dist, int cd) {
	int c;

	lcdc_io[1].state = lcdc_io[2].state = 0;
	portc_io[7].state = portc_io[6].state = portc_io[5].state = 0;
	portc_io[4].state = 1;
	portc_io[3].state = cd;
	portc_io[2].state = (dist >= 0)?1:0;
	write_pins(lcdc_io, 3);
	write_pins(portc_io, 8);

	dist = abs(dist);

	for (c = 0; c < dist; c++)
		lcd_execute(40);
}

void print_lcd(char *string) {
	int c, d;

	lcdc_io[1].state = 1;
	lcdc_io[2].state = 0;

	write_pins(lcdc_io, 3);

	for (c = 0; string[c] != '\0'; c++) {
		switch (string[c]) {
			case '\n':
				set_lcd_cursor(0, 1);
				break;

			default:
				for (d = 0; d < 8; d++)
					portc_io[d].state = (string[c] >> d) & 1;

				write_pins(portc_io, 8);
				lcd_execute(40);
				break;
		}
	}
}

