#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <errno.h>

#include "stk.h"
#ifdef USE_IOBOARD
#include "ioboard.h"
#else
#include <curses.h>
#endif

#define DISPLAY_LENGTH 16

#define FRAME_TIME 100000 /* In microseconds */
#define UPDATE_TIME 5 /* Frames between updates */

#ifndef USE_IOBOARD
#define BUGGY_GLYPH ''
#define BUGGY_JUMP_GLYPH ''
#define FLAT_TERRAIN_GLYPH '_'
#define CRATER_GLYPH ' '
#define BOULDER_GLYPH '.'
#else
#define BUGGY_GLYPH 4
#define BUGGY_JUMP_GLYPH 3
#define FLAT_TERRAIN_GLYPH 2
#define CRATER_GLYPH 1
#define BOULDER_GLYPH 0
#endif

/* Protocol commands */
#define UNKNOWN_CMD 0
#define JOIN_CMD 1
#define WLCM_CMD 2
#define REDY_CMD 3
#define STRT_CMD 4
#define SCOR_CMD 5
#define DEAD_CMD 6
#define OVER_CMD 7
#define QUIT_CMD 8
#define EROR_CMD 10

void cleanup(void);
int start_game(int seed);
void get_keys(unsigned int *, unsigned int *, unsigned int *);
void update(char *, unsigned int, unsigned int, unsigned int);
void draw(char *buffer);
long wait_for_start(void);
int join_server(char *name);
int get_winner(char *name, int length);
void print_won(int score);
void print_lost(char *name, int score);
unsigned long utime(void);
int get_command(char *buffer, int length, char **arg1, char **arg2);
void ssleep(unsigned long);

char charset[] = {
 	0x00, 0x0E, 0x1F, 0x1E, 0x0E, 0x1F, 0x16, 0x0D, /* boulder */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x1B, 0x1F, /* crater */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1F, 0x16, 0x0D, /* flat terrain */
	0x00, 0x0C, 0x1F, 0x1F, 0x0A, 0x00, 0x00, 0x00, /* jumping buggy */
	0x00, 0x0C, 0x1F, 0x1F, 0x0A, 0x1F, 0x16, 0x0D  /* buggy */
};

int sckt;

int main(int argc, char **argv) {
	int score, win_score, port = 6544;
	int seed;
	char *hostname;
	char *player, winner[16];
	printf("1\n");
	if (argc < 3) {
		printf("Usage:\n\tmbgy <playername> <hostname> [port]\n");
		exit(EXIT_FAILURE);
	}
printf("2\n");
	player = argv[1];
	hostname = argv[2];

	if (argc > 3)
		port = atoi(argv[3]);

	atexit(cleanup);
printf("3\n");
#ifndef USE_IOBOARD
	/* Send error messages somewhere useful */
	freopen("mbgy_log.txt", "w", stderr);

	/* Set up curses */
	initscr();
	nonl();
	cbreak();
	noecho();
	keypad(stdscr, TRUE);
	nodelay(stdscr, 1);
	printf("4\n");
	/* Flush input buffer */
	while (getch() != ERR);
#else
	open_ioboard();

	/* Display on, cursor off, shift cursor right, no display shift on write */
	set_lcd(1, 0, 1, 0);
printf("5\n");
	lcd_charset(charset, 5);
#endif

	sckt = stk_connect(hostname, port, SOCK_DGRAM);

	if (sckt < 0) {
		fprintf(stderr, "Error opening socket\n");
		exit(EXIT_FAILURE);
	}
printf("6\n");
	if (join_server(player) == -1) {
		fprintf(stderr, "Error connecting to server\n");
		exit(EXIT_FAILURE);
	}
printf("7\n");
	for (;;) {
		/* Wait for the game to start */
		if ((seed = wait_for_start()) == -1) {
			fprintf(stderr, "Error reading from socket (wait_for_start)\n");
			exit(EXIT_FAILURE);
		}
printf("8\n");
		if ((score = start_game(seed)) == -1) {
			fprintf(stderr, "Error reading from socket (start_game)\n");
			exit(EXIT_FAILURE);
		}
printf("9\n");
		if ((win_score = get_winner(winner, 16)) == -1) {
			fprintf(stderr, "Error reading from socket (get_winner)\n");
			exit(EXIT_FAILURE);
		}
printf("10\n");
		if (score == win_score)
			print_won(score);
		else
			print_lost(winner, win_score);
printf("11\n");
		sleep(3);
	}

	cleanup();
printf("12\n");
	return 0;
}

void cleanup(void) {
	if (sckt >= 0)
		close(sckt);

#ifndef USE_IOBOARD
	/* Flush input buffer */
	while (getch() != ERR);

	endwin();
#else
	clear_lcd();
#endif
}

int start_game(int seed) {
	int c =0;
	unsigned int buggy_pos = (DISPLAY_LENGTH / 4) * 3, buggy_jump_time = 0;
	unsigned int play_time = 0, flat_length = 0, holddown = 0, dead = 0;
	unsigned int update_timer = UPDATE_TIME;
	char next_char, buffer[256];
	unsigned long frame_start_time;

	unsigned int left = 0, right = 0, up = 0;

	/* Display Buffer */
	char dbuf[DISPLAY_LENGTH + 1];
printf("gs0\n");
//	memset(dbuf, FLAT_TERRAIN_GLYPH, DISPLAY_LENGTH);
	for(c = 0; c <= DISPLAY_LENGTH; c++)
	{
		dbuf[c]=FLAT_TERRAIN_GLYPH;
	}
	dbuf[DISPLAY_LENGTH] = '\0';
//printf("gs1 seed = %i\n", seed);
//	srand(seed); /* Set the random seed */
printf("gs2\n");
	/* Game loop */
	while (!dead) {
		/* Record current time */
		frame_start_time = utime();
printf("gs2.1\n");
		/* Move forward by one space */
		if (!flat_length) { /* Time for an obstacle! */
			if (rand() > RAND_MAX / 2)
				next_char = CRATER_GLYPH;
			else
				next_char = BOULDER_GLYPH;

			flat_length = (rand() % 10) + 4;
		}
		else
			next_char = FLAT_TERRAIN_GLYPH;
printf("gs2.2\n");

		memmove(dbuf + 1, dbuf, DISPLAY_LENGTH - 1);
		dbuf[0] = next_char;
printf("gs3\n");
		/* Check for collisions */
		if (!buggy_jump_time && (dbuf[buggy_pos] != FLAT_TERRAIN_GLYPH))
			dead = 1; /* Player hit something! */

		/* Get control key states */
		get_keys(&left, &right, &up);

		if (up && !buggy_jump_time && !holddown) { /* Jump! */
			buggy_jump_time = 4;
			up = 0;
		}

		if (left && (buggy_pos > 0)) {
			buggy_pos--;
			left = 0;
		}

		if (right && (buggy_pos < (DISPLAY_LENGTH - 1))) {
			buggy_pos++;
			right = 0;
		}

		/* Draw the display */
		snprintf(buffer, (2 * DISPLAY_LENGTH) + 3, "%-16i\n%s\n", (play_time * FRAME_TIME / 1000000), dbuf);
		buffer[buggy_pos + (buggy_jump_time?0:1 * 17)] = BUGGY_GLYPH;

		draw(buffer);

		if (holddown)
			holddown = 0;

		if (buggy_jump_time) {
			buggy_jump_time--;

			if (!buggy_jump_time) /* Buggy has landed, set holddown timer */
				holddown = 1;
		}

		play_time++;
		flat_length--;

		/* Is it time to send our score to the server? */
		if (update_timer == 0) {
			update_timer = UPDATE_TIME;
			snprintf(buffer, 256, "SCOR %i\n", play_time);
			if (send(sckt, buffer, strlen(buffer) + 1, 0) == -1)
				return -1;
		}
		else
			update_timer--;

		/* Wait until FRAME_TIME has elapsed */
		ssleep(FRAME_TIME - (utime() - frame_start_time));
	}

	/* Tell server we're dead */
	snprintf(buffer, 256, "DEAD %i\n", play_time);
	if (send(sckt, buffer, strlen(buffer) + 1, 0) == -1)
		return -1;

	snprintf(buffer, 256, "    You Died!   \nYour Score: %4i", play_time);
	draw(buffer);

	return play_time;
}

int get_winner(char *name, int length) {
	char buffer[256], *winner, *score;

	if (get_command(buffer, 256, &winner, &score) != OVER_CMD) {
		send(sckt, "EROR\n", 5, 0);
		return -1;
	}

	strncpy(name, winner, length);

	return atoi(score);
}

void draw(char *buffer) {
#ifdef USE_IOBOARD
	print_lcd(buffer);
#else
	mvprintw(0, 0, buffer);
	refresh();
#endif
}

void get_keys(unsigned int *left, unsigned int *right, unsigned int *up) {
#ifdef USE_IOBOARD
	get_buttons(left, up, right);
#else
	int c;

	while ((c = getch()) != ERR) {
		switch (c) {
			case KEY_LEFT: *left = 1; break;
			case KEY_RIGHT: *right = 1; break;
			case KEY_UP: *up = 1; break;
			default: return;
		}
	}
#endif
}

int join_server(char *name) {
	char buffer[256];

	/* Send JOIN */
	snprintf(buffer, 256, "JOIN %s\n", name);
	if (send(sckt, buffer, strlen(buffer) + 1, 0) == -1)
		return -1;

	/* Get WLCM */
	if (get_command(buffer, 256, NULL, NULL) != WLCM_CMD) {
		send(sckt, "EROR\n", 5, 0);
		return -1;
	}

	return 0;
}

long wait_for_start(void) {
	char buffer[256], *atime, *seed;
	int command = 0;
	long start_time;
printf("wfs1\n");
	/* Tell the user what's going on */
	snprintf(buffer, 256, "  Waiting for  \n game start... ");
	draw(buffer);
printf("wfs2\n");
	if (get_command(buffer, 256, &atime, NULL) != REDY_CMD) {
		send(sckt, "EROR\n", 5, 0);
		return -1;
	}
printf("wfs3\n");
	/* How long until the game is expected to start? */
	start_time = atol(atime) + utime();
printf("wfs4\n");
	/* Go non-blocking */
	fcntl(sckt, F_SETFL, fcntl(sckt, F_GETFL) | O_NDELAY);
printf("wfs5\n");
	/* Print a countdown timer until the game starts */
	while (command == 0) {
		if (start_time > utime())
			snprintf(buffer, 256, "  Waiting for  \n  players... %2li \n", (start_time - utime()) / 1000000);
		else
			snprintf(buffer, 256, "  Waiting for  \n  players...\n");
printf("wfs5.1\n");
		draw(buffer);
printf("wfs5.2\n");
		/* Check for STRT */
		if ((command = get_command(buffer, 256, &seed, NULL)) == -1)
			return -1;
printf("wfs5.3\n");
		/* Nothing recved yet */
		if (command == 0) {
			sleep(1);
			continue;
		}
printf("wfs5.4\n");
		if (command != STRT_CMD) {
			send(sckt, "EROR\n", 5, 0);
			return -1;
		}
	}
printf("wfs6\n");
	/* Return to Blocking mode */
	fcntl(sckt, F_SETFL, fcntl(sckt, F_GETFL) ^ O_NDELAY);
printf("wfs7\n");
	return atol(seed);
}

void print_won(int score) {
	char buffer[256];

	snprintf(buffer, 256, "    You Won!    \nYour Score: %4i", score);
	draw(buffer);
}

void print_lost(char *name, int score) {
	char buffer[256];

	snprintf(buffer, 256, "    You Lost!   \n%5s Won: %5i", name, score);
	draw(buffer);
}

unsigned long utime(void) {
	struct timeval tv;
	struct timezone tz;

	gettimeofday(&tv, &tz);

	return (tv.tv_sec * 1000000) + tv.tv_usec;
}

int get_command(char *buffer, int length, char **arg1, char **arg2) {
	int data_length;
	char *c, *t1 = NULL, *t2 = NULL;

	data_length = recv(sckt, buffer, length - 1, 0);

	if (data_length == -1) {
		if (errno == EWOULDBLOCK)
			return 0;
		else
			return -1;
	}

	/* Guarantee null-termination */
	buffer[data_length] = '\0';

	for (c = buffer; *c != '\0'; c++) {
		/* If this char is a space, and next is not the end of the command */
		if ((*(c + 1) != '\0') && (*c == ' ')) {
			if (t1 == NULL)
				t1 = c + 1;
			else if (t2 == NULL)
				t2 = c + 1;
			else /* We've found both arguments already */
				break;
			*c = '\0';
		}
		if (*c == '\n') /* Remove \n */
			*c = '\0';
	}

	/* Only return arguments that will be used */
	if (arg1 != NULL)
		*arg1 = t1;
	if (arg2 != NULL)
		*arg2 = t2;

	if (strcmp("WLCM", buffer) == 0)
		return WLCM_CMD;
	else if (strcmp("REDY", buffer) == 0)
		return REDY_CMD;
	else if (strcmp("STRT", buffer) == 0)
		return STRT_CMD;
	else if (strcmp("OVER", buffer) == 0)
		return OVER_CMD;
	else if (strcmp("EROR", buffer) == 0)
		return EROR_CMD;
	else if (strcmp("QUIT", buffer) == 0)
		return QUIT_CMD;
	else
		return UNKNOWN_CMD;
}

void ssleep(unsigned long wait) {
	unsigned long done_time = wait + utime();

	while (utime() < wait);
}

