#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <errno.h>

#include "stk.h"
#include "mbgyd.h"

/*
 * mbgyd: The mbgy daemon
 *
 * Protocol: (All messages are UDP datagrams)
 *  - Clients connect to server by sending a JOIN command
 *  - Server replies with WLCM command
 *  - Server starts game by sending STRT command
 *  - Clients report score to server by sending SCOR command
 *  - Clients report player death by sending DEAD command
 *  - Server ends game by sending OVER command
 *
 *  JOIN:
 *    JOIN <playername>\n
 *
 *  WLCM:
 *    WLCM\n
 *
 *    Acknowledge JOIN
 *
 *  REDY:
 *    REDY <starttime>\n
 *
 *    starttime is the estimated time in microseconds until the game will begin.
 *    0 means the time is unknown. Clients must wait for the STRT command
 *    before starting!
 *
 *  STRT:
 *    STRT <seed>\n
 *
 *    seed is the random seed of the level generator. All clients will recieve
 *    the same seed for a given game.
 *
 *  SCOR:
 *    SCOR <score>\n
 *
 *    score is the player's current score. This command will be issued to the
 *    server by the client once every second.
 *
 *  DEAD:
 *    DEAD <score>\n
 *
 *    score is the player's score at the time of death.
 *
 *  OVER:
 *    OVER <winnername> <score>\n
 *
 *    winnername is the name of the winner. score is the winner's score.
 *    The recieving client may only assume it is the winner if the score
 *    matches it's own, as duplicate player names are allowed.
 *
 *  QUIT:
 *    QUIT
 *
 *    Server closing, or player leaving game.
 *
 *  EROR:
 *    EROR
 *
 *    This command may be issued by either server or client at any time
 *    if the issuer recieves unexpected or otherwise erroneous data from
 *    the issuee.
 */

#define UNKNOWN_CMD 0
#define JOIN_CMD 1
#define WLCM_CMD 2
#define REDY_CMD 3
#define STRT_CMD 4
#define SCOR_CMD 5
#define DEAD_CMD 6
#define OVER_CMD 7
#define QUIT_CMD 8
#define EROR_CMD 10

#define MIN_PLAYERS 2 /* Default minimum players */
#define MAX_PLAYERS 4 /* Default maximum players */
#define JOIN_DELAY 15 /* Default time (s) to wait for more players after minimum players have joined */
#define PORT 6544
#define PAGE_DELAY 5 /* Interval of html generation, in secs */
#define STATUS_PAGE "/home/phil/public_html/mbgy.html"
#define POLL_DELAY 10000 /* Time in microseconds to delay between polling the socket */
#define DISCONNECT_TIME 1500000 /* in microseconds */

int wait_for_players(void);
int new_player(char *name, struct sockaddr_in *addr);
int play_game(void);
int find_winner(void);
int find_player(struct sockaddr_in *addr);
int get_command(char *buffer, int length, char **arg1, char **arg2, struct sockaddr_in *addr);
unsigned long utime(void);
void ssleep(unsigned long wait);

int sckt;
int numplayers = 0, minplayers = MIN_PLAYERS, maxplayers = MAX_PLAYERS, join_delay = JOIN_DELAY;
int numgames = 0; /* Number of completed games */
struct player players[MAX_PLAYERS];

int main(int argc, char **argv) {
	/* Read in limits from the command-line */
	if (argc > 1)
		minplayers = atoi(argv[1]);
	if (argc > 2)
		maxplayers = atoi(argv[2]);
	if (argc > 3)
		join_delay = atoi(argv[3]);

	/* Verify that the limits are sane */
	if (maxplayers > MAX_PLAYERS)
		maxplayers = MAX_PLAYERS;
	if (minplayers > maxplayers)
		minplayers = maxplayers;
	if (minplayers < 1)
		minplayers = 1;

	sckt = stk_listen(NULL, PORT, SOCK_DGRAM, 5); /* Any address, port 6544, UPD, backlog 5 */

	if (sckt < 0) {
		fprintf(stderr, "Error opening listening socket: %i", sckt);
		return 1;
	}

	for (;;) {
		/* Start a new game */
		if (wait_for_players() == -1) {
			fprintf(stderr, "Socket Error\n");
			close(sckt);
			exit(EXIT_FAILURE);
		}
		if (play_game() == -1) {
			fprintf(stderr, "Socket Error\n");
			close(sckt);
			exit(EXIT_FAILURE);
		}
	}

	return 0;
}

int wait_for_players(void) {
	struct sockaddr_in addr;
	int c, command, addr_len = sizeof(struct sockaddr);
	unsigned long time_to_start = 0;
	char buffer[256], *name;

	printf("Waiting for new players (%i players already in game)...\n", numplayers);

	/* Wait for new players */
	while (numplayers < minplayers) {
		if ((command = get_command(buffer, 256, &name, NULL, &addr)) == -1)
			return -1; /* Socket error */

		if (command == JOIN_CMD)
			new_player(name, &addr);
		else {
			/* Wrong command. Issue error. */
			if (sendto(sckt, "EROR\n", 5, 0, (struct sockaddr *) &addr, addr_len) == -1)
				return -1;
		}
	}

	/* Okay, time to start counting down to game start */
	time_to_start = utime() + (join_delay * 1000000);

	printf("Game will start in %i seconds...\n", join_delay);

	/* Tell players we're REDY for a new game */
	for (c = 0; c < numplayers; c++) {
		snprintf(buffer, 256, "REDY %li\n", time_to_start - utime());
		if (sendto(sckt, buffer, strlen(buffer) + 1, 0, (struct sockaddr *) &players[c].addr, addr_len) == -1)
			return -1;
	}

	/* Go non-blocking */
	fcntl(sckt, F_SETFL, fcntl(sckt, F_GETFL) | O_NDELAY);

	/* Wait for more new players */
	while ((numplayers < maxplayers) && (utime() < time_to_start)) {
		if ((command = get_command(buffer, 256, &name, NULL, &addr)) == -1)
			return -1; /* Socket error */

		/* Did we get a command? */
		if (command == 0) {
			ssleep(POLL_DELAY);
			continue;
		}

		if (command == JOIN_CMD) {
			new_player(name, &addr);

			/* Let the player know we're REDY */
			snprintf(buffer, 256, "REDY %li\n", time_to_start - utime());
			if (sendto(sckt, buffer, strlen(buffer) + 1, 0, (struct sockaddr *) &players[c].addr, addr_len) == -1)
				return -1;
		}
		else {
			/* Wrong command. Issue error. */
			if (sendto(sckt, "EROR\n", 5, 0, (struct sockaddr *) &addr, addr_len) == -1)
				return -1;
		}
	}

	/* Return to blocking */
	fcntl(sckt, F_SETFL, fcntl(sckt, F_GETFL) ^ O_NDELAY);

	return 1;
}

int new_player(char *name, struct sockaddr_in *addr) {
	int c, addr_len = sizeof(struct sockaddr);
	char buffer[256];

	/* Check that this player hasn't already joined the game,
	   and that we have room for one more player */
	if ((numplayers < maxplayers) && (find_player(addr) == -1)) {
		c = numplayers;
		numplayers++;

		/* Copy name and address of player */
		strncpy(players[c].name, name, 16);
		memcpy(&players[c].addr, addr, sizeof(struct sockaddr));

		/* Initialize the player to zero */
		players[c].dead = players[c].score = 0;
		players[c].high = players[c].low = players[c].avg = 0;
		players[c].gamesplayed = 0;
		players[c].connected = 1;

		printf("%s (%s:%i) has joined the game\n", players[c].name, inet_ntoa(players[c].addr.sin_addr), players[c].addr.sin_port);

		/* Player is in the game, send acknowledge */
		snprintf(buffer, 256, "WLCM\n");
		if (sendto(sckt, buffer, strlen(buffer) + 1, 0, (struct sockaddr *) &players[c].addr, addr_len) == -1)
			return -1;

		return 1;
	}
	else {
		sendto(sckt, "EROR\n", 6, 0, (struct sockaddr *) &addr, addr_len);
		return -1;
	}
}

int play_game(void) {
	struct sockaddr_in addr;
	int addr_len = sizeof(struct sockaddr);
	int command, winner, game_not_over = numplayers;
	unsigned int c;
	long seed;
	unsigned long page_time = utime() + (PAGE_DELAY * 1000000);
	char buffer[256], *score;

	seed = time(NULL);

	/* Increment gamesplayed */
	for (c = 0; c < numplayers; c++)
		if (players[c].connected)
			players[c].gamesplayed++;

	printf("Game starting with %i players.\n", numplayers);

	/* Send all players the game seed */
	snprintf(buffer, 256, "STRT %li\n", seed);
	for (c = 0; c < numplayers; c++)
		if (sendto(sckt, buffer, strlen(buffer) + 1, 0, (struct sockaddr *) &players[c].addr, addr_len) == -1)
			return -1;

	/* Go non-blocking */
	fcntl(sckt, F_SETFL, fcntl(sckt, F_GETFL) | O_NDELAY);

	/* While the game isn't over */
	while (game_not_over) {
		/* Write web status page, if it's time */
		if (utime() > page_time) {
			htmlgen(STATUS_PAGE, players, numplayers);
			page_time = utime() + (PAGE_DELAY * 1000000);
		}

		command = get_command(buffer, 256, &score, NULL, &addr);

		if (command == -1)
			return -1;

		if (command == 0) { /* Nothing recved yet. */
			ssleep(POLL_DELAY); /* 1/10th the default client frame_time */
			continue;
		}

		if ((c = find_player(&addr)) == -1) {
			/* Are you joining? */
			if (command == JOIN_CMD) {
				/* Welcome. We'll start in a little while */
				new_player(score, &addr); /* Funny thing, score == name */
			}
			else {
				/* Who are you, and what are you doing here? */
				if (sendto(sckt, "EROR\n", 6, 0, (struct sockaddr *) &addr, addr_len) == -1)
					return -1;
			}
		}
		else {
			switch (command) {
				case SCOR_CMD:
					/* Buffer overflow alert? */
					players[c].score = atoi(score);
					players[c].lastupdate_time = utime();
					break;
				case DEAD_CMD:
					players[c].score = atoi(score);
					players[c].dead = 1;
					players[c].lastupdate_time = utime();
					printf("%s died with score %i\n", players[c].name, players[c].score);
					game_not_over--;
					break;
				default:
					/* Huh? */
					if (sendto(sckt, "EROR\n", 5, 0, (struct sockaddr *) &addr, addr_len) == -1)
						return -1;
					break;
			}
		}

		/* Check that players are still connected */
		for (c = 0; c < numplayers; c++) {
			if (players[c].gamesplayed && !players[c].dead && players[c].connected && (players[c].lastupdate_time < (utime() - DISCONNECT_TIME))) {
				players[c].connected = 0;
				game_not_over--;
				printf("%s has left the game.\n", players[c].name);
			}
		}
	}

	/* Return to blocking */
	fcntl(sckt, F_SETFL, fcntl(sckt, F_GETFL) ^ O_NDELAY);

	/* Figure out who won */
	winner = find_winner();

	printf("Game over. %s is the winner with score %i\n", players[winner].name, players[winner].score);

	/* GAME OVER! Tell players who won */
	snprintf(buffer, 256, "OVER %s %i\n", players[winner].name, players[winner].score);
	for (c = 0; c < numplayers; c++)
		if (players[c].gamesplayed && players[c].connected)
			if (sendto(sckt, buffer, strlen(buffer) + 1, 0, (struct sockaddr *) &players[c].addr, addr_len) == -1)
				return -1;

	/* Calculate stats and prepare for next game */
	for (c = 0; c < numplayers; c++) {
		if (players[c].gamesplayed && players[c].connected) {
			if (players[c].score > players[c].high)
				players[c].high = players[c].score;
			if ((players[c].score < players[c].low) || (players[c].low == 0))
				players[c].low = players[c].score;
			players[c].avg = ((players[c].gamesplayed * players[c].avg) + players[c].score) / (players[c].gamesplayed);

			players[c].score = players[c].dead = 0;
		}
	}

	/* Write web status page one more time */
	htmlgen(STATUS_PAGE, players, numplayers);

	return 1;
}

int find_winner(void) {
	int c, winner = 0;

	for (c = 0; c < numplayers; c++) {
		if (players[c].gamesplayed && players[c].connected && (players[c].score > players[winner].score))
			winner = c;
	}

	return winner;
}

int find_player(struct sockaddr_in *addr) {
	int addr_len = sizeof(struct sockaddr), c;

	for (c = 0; c < numplayers; c++) {
		if (memcmp(addr, &players[c].addr, addr_len) == 0)
			break;
	}

	if (c == numplayers) /* Player not found */
		return -1;
	else
		return c;
}

int get_command(char *buffer, int length, char **arg1, char **arg2, struct sockaddr_in *addr) {
	unsigned int addr_len = sizeof(struct sockaddr);
	int data_length;
	char *c, *t1 = NULL, *t2 = NULL;

	data_length = recvfrom(sckt, buffer, length - 1, 0, (struct sockaddr *) addr, &addr_len);

	if (data_length == -1) {
		if (errno == EWOULDBLOCK)
			return 0;
		else
			return -1;
	}

	/* Guarantee null-termination */
	buffer[data_length] = '\0';

	for (c = buffer; *c != '\0'; c++) {
		/* If this char is a space, and next is not the end of the command */
		if ((*(c + 1) != '\0') && (*c == ' ')) {
			if (t1 == NULL)
				t1 = c + 1;
			else if (t2 == NULL)
				t2 = c + 1;
			else /* We've found both arguments already */
				break;
			*c = '\0';
		}
		if (*c == '\n') /* Remove \n */
			*c = '\0';
	}

	/* Only return arguments that will be used */
	if (arg1 != NULL)
		*arg1 = t1;
	if (arg2 != NULL)
		*arg2 = t2;

	if (strcmp("JOIN", buffer) == 0)
		return JOIN_CMD;
	else if (strcmp("SCOR", buffer) == 0)
		return SCOR_CMD;
	else if (strcmp("DEAD", buffer) == 0)
		return DEAD_CMD;
	else if (strcmp("EROR", buffer) == 0)
		return EROR_CMD;
	else if (strcmp("QUIT", buffer) == 0)
		return QUIT_CMD;
	else
		return UNKNOWN_CMD;
}

unsigned long utime(void) {
	struct timeval tv;
	struct timezone tz;

	gettimeofday(&tv, &tz);

	return (tv.tv_sec * 1000000) + tv.tv_usec;
}

void ssleep(unsigned long wait) {
	unsigned long done_time = wait + utime();

	while (utime() < wait);
}

