/*
 * stk.c: Socket Toolkit
 * Copyright (c) 2003 Philip Bock
 */

#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "stk.h"

struct sockaddr *stk_address(const char *hostname, int port)
{
	struct hostent *host;
	struct sockaddr_in *address;

	if (hostname != NULL)
		if ((host = gethostbyname(hostname)) == NULL)
			return NULL;

	address = malloc(sizeof(struct sockaddr_in));

	address->sin_family = AF_INET;
	address->sin_port = htons(port);

	if (hostname != NULL)
		memcpy(&address->sin_addr, host->h_addr_list[0], host->h_length);
	else
		address->sin_addr.s_addr = INADDR_ANY;

	memset(&address->sin_zero, '\0', 8);

	return (struct sockaddr *) address;
}

int stk_connect(const char *hostname, int port, int type)
{
	struct sockaddr *address;
	int sckt;

	if ((address = stk_address(hostname, port)) == NULL)
		return -2;

	if ((sckt = socket(PF_INET, type, 0)) == -1)
		return -1;

	if (connect(sckt, address, sizeof(struct sockaddr)) == -1) {
		close(sckt);
		return -3;
	}

	free(address);

	return sckt;
}

int stk_listen(const char *ip, int port, int type, int backlog)
{
	struct sockaddr *address;
	int sckt;

	if ((address = stk_address(ip, port)) == NULL)
		return -2; /* Error resolving hostname */

	if ((sckt = socket(PF_INET, type, 0)) == -1)
		return -1; /* Error creating socket */

	if (bind(sckt, address, sizeof(struct sockaddr)) == -1) {
		close(sckt);
		return -3; /* Error binding to port */
	}

	free(address);

	if (type == SOCK_STREAM) {
		if (listen(sckt, backlog) == -1) {
			close(sckt);
			return -4; /* Error listening on socket */
		}
	}

	return sckt;
}

